/**************************************************************************/
/* FILE NAME: FFT_freq.c               COPYRIGHT (c) Freescale 2006 	  */
/*                                                All Rights Reserved     */
/* DESCRIPTION:                                                           */
/* This file contains the necessary setup code for configuring the eMIOS  */
/* for GPIO mode on CH0 and OPWM mode on CH5.							  */
/*========================================================================*/
/* REV      AUTHOR       DATE       DESCRIPTION OF CHANGE                 */
/* ---   -----------   ----------   ---------------------                 */
/* 0.1	 R. Moran		Jan 07		Initial Version				          */
/**************************************************************************/ 

#include "libdsp2.h"
#include <spe.h>
#include "vars.h"

// Sampling frequency = ADC CLK / (14+2) for Single Ended Conversions
#define SAMPLE_FREQ 367

// Set threshold so that only FFT results above this value are compared
#define Threshold 1000000

int	 real_data[1024];
int imag_data[1024];
int value1;
int value2;

extern int highest_freqx;


/*************************************************************************
*	Function Name: 	analyse_FFT											 *
*	Description: 	This function takes the FFT output and performs		 *
*					analsyis on it to determine the frequency. The 	     *
*					magnitude of both the real and imaginary part are	 *
*					analysed for an accurate frequency.	This function    *
*					returns the FFT sample point and magnitude of the	 *
*					Frequency detected.									 *
*						                                                 *
*	Called By :		FFT_freq											 *
**************************************************************************/ 
int analyse_FFT(void)
{
	int i = 0;
	int Freq_peak = 0;
	int highest_peak[2] = {0,0};
	
	__ev64_opaque__ peak_threshold;
	__ev64_opaque__ FFT_data;
	
	// Create threshold point for detecting freq components
	// This reduces the amount of FFT points that have to be compared.
	// This threshold can be reduced in the #define "Threshold" if inaccurate results 
	// of "0" appear.
	peak_threshold = __ev_create_s32(Threshold,Threshold);
	

	// Find highest Freq component of both imaginary results and real results
	
	// Loop for 512 results of imaginary results
	for(i=0; i<512; i=i+2)
	{
		// Create 64 bit vector comprising of real FFT data
		FFT_data = __ev_create_s32(imag_data[i], imag_data[i+1]);
		
		// Compare 64bit FFT data against freq peak threshold
		Freq_peak = __ev_any_gts(FFT_data,peak_threshold);
		
		// If freq peak detected then branch to "Freq_detected"
		if (Freq_peak)
		{
			// The following logic is used to detect the largest freq component from the
			// FFT data. It will store this value in "highest_peak[0]" and the corresponding
			// sample index in "highest_peak[1]"
			if((imag_data[i] > imag_data[i+1]) && (imag_data[i] > highest_peak[0]))
			{
				highest_peak[0] = imag_data[i];
				highest_peak[1] = i;
			}
			else if ((imag_data[i+1] > imag_data[i]) && (imag_data[i+1] > highest_peak[0]))
			{
				highest_peak[0] = imag_data[i+1];
				highest_peak[1] = i+1;
			}
		}
	}
	
	// Loop for 511 results of real results (omit point 0)
	for(i=1; i<512; i=i+2)
	{
		// Create 64 bit vector comprising of real FFT data
		FFT_data = __ev_create_s32(real_data[i], real_data[i+1]);
		
		// Compare 64bit FFT data against freq peak threshold
		Freq_peak = __ev_any_gts(FFT_data,peak_threshold);
		
		// If freq peak detected then branch to "Freq_detected"
		if (Freq_peak)
		{
			// The following logic is used to detect the largest freq component from the
			// FFT data. It will store this value in "highest_peak[0]" and the corresponding
			// sample index in "highest_peak[1]"
			if((real_data[i] > real_data[i+1]) && (real_data[i] > highest_peak[0]))
			{
				highest_peak[0] = real_data[i];
				highest_peak[1] = i;
			}
			else if ((real_data[i+1] > real_data[i]) && (real_data[i+1] > highest_peak[0]))
			{
				highest_peak[0] = real_data[i+1];
				highest_peak[1] = i+1;
			}
		}
	}
		

	return(highest_peak[1]);
}


/*************************************************************************
*	Function Name: 	FFT_freq											 *
*	Description: 	This functions calls the SPE FFT function and then   *
*					converts the results into a Frequency. Where possible*
*					SPE is used to improve performance.					 *						                                                 *
*	Called By :		FFT_freq											 *
**************************************************************************/ 
int FFT_freq(int inout_bufferx[2048]) {
	
	int i,j = 0;
	int highest_freq_component = 0;
	int highest_freqx = 0;
	float sample_size;
	
	
	__ev64_opaque__ SPE_data;
	__ev64_opaque__ abs_data;
	
	
	// Enable SPE
	asm("mfmsr r3");
	asm("lis r4, 0x0200");
	asm("or r3,r4,r3");
	asm("mtmsr r3");
    
	
		
	/* bit reverse permutation and calculation of 256-point FFT */
	bitrev_table_16bit(16, (short *)(inout_bufferx+1024), seed_radix4_16bit_1024);
	fft_radix4_frac32 (1024, inout_bufferx, w_table_radix4_frac32_1024);
	
	for (i=0; i<2047; i=i+2)
	{
		// Create 64bit vector for each imag/real pair
		SPE_data = __ev_create_s32(inout_bufferx[i], inout_bufferx[i+1]);
		
		// Find the absolute values of the FFT results
		abs_data = __ev_abs(SPE_data);
		
		// Split the absolute values into Real/Imaginary Parts
		real_data[j] = __ev_get_upper_s32(abs_data); 
		imag_data[j] = __ev_get_lower_s32(abs_data);
		j++;
	}
	
	// Get largest frequency component
	highest_freq_component = analyse_FFT();
	
	// Find frequency of sample step - (Sample Freq/2) * (1/No. Sample points)
	sample_size = (0.5*SAMPLE_FREQ)*0.00195312;
	
	// Find frequency of largest frequency component - sample * sample step freq
	highest_freqx = sample_size * highest_freq_component;
	
	return highest_freqx;
		
}
